<?php

namespace Application\Services\Mail\Cron;

use Application\Model\EmailTemplate;
use Application\Model\GuideAssignment;
use Application\Model\GuideAssignmentTable;
use Application\Model\TaskAssignment;
use Application\Model\TaskAssignmentTable;
use Application\Model\TaskType;
use Application\Model\TaskTypesTable;
use Application\Model\TourType;
use Application\Model\TourTypesTable;
use Application\Model\User;
use Application\Model\UserTable;
use Application\Services\Mail\GenericMailerInterface;
use Laminas\Db\ResultSet\ResultSet;
use Laminas\View\Model\ViewModel;

class TaskScheduleFutureMailer extends GenericMailerInterface
{
    /**
     * @var User[]
     */
    private $usersList = null;

    /**
     * @var TaskType[]
     */
    private $taskTypeList = null;

    /**
     * @var TourType[]
     */
    private $tourTypeList = null;

    /**
     * @return User[]|null
     */
    private function getUsers()
    {
        if ($this->usersList === null)
        {
            $this->usersList = [];

            /**
             * @var $usersTable UserTable
             */
            $usersTable = $this->serviceLocator->get('Application\Model\UserTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $usersTable->getLiveUsers();

            /**
             * @var $user User
             */
            while ($user = $resultSet->current())
            {
                $this->usersList[$user->id] = $user;
                $resultSet->next();
            }

        }

        return $this->usersList;
    }

    /**
     * @param $id
     * @return User
     */
    private function getUser($id)
    {
        $usersList = $this->getUsers();

        if (array_key_exists($id, $usersList) && $usersList[$id] instanceof User)
        {
            return $usersList[$id];
        }

        $noUser = new User();
        $noUser->name = sprintf('Unknown User (id: %d)', $id);
        return $noUser;
    }

    /**
     * @param TaskAssignment $taskAssignment
     * @return User[]
     */
    private function getAssignedGuides(TaskAssignment $taskAssignment)
    {

        /**
         * @var $guideAssignmentsTable GuideAssignmentTable
         */
        $guideAssignmentsTable = $this->serviceLocator->get('Application\Model\GuideAssignmentTable');

        /**
         * @var $guideAssignmentsResultSet ResultSet
         */
        $guideAssignmentsResultSet = $guideAssignmentsTable->fetchAllByAssignmentIds([$taskAssignment->id], 1);

        $assigned_guides = [];

        /**
         * @var $guideAssignment GuideAssignment
         */
        while ($guideAssignment = $guideAssignmentsResultSet->current())
        {
            $assigned_guides[] = $this->getUser($guideAssignment->guide_id);
            $guideAssignmentsResultSet->next();
        }

        return $assigned_guides;
    }

    /**
     * @return TaskType[]
     */
    private function getTaskTypeList()
    {
        if ($this->taskTypeList == null)
        {
            $this->taskTypeList = [];

            /**
             * @var $taskTypeTable TaskTypesTable
             */
            $taskTypeTable = $this->serviceLocator->get('Application\Model\TaskTypeTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $taskTypeTable->fetchAll();

            /**
             * @var $taskType TaskType
             */
            while ($taskType = $resultSet->current())
            {
                $this->taskTypeList[$taskType->id] = $taskType;
                $resultSet->next();
            }
        }

        return $this->taskTypeList;
    }

    /**
     * @param $id
     * @return TaskType
     */
    private function getTaskType($id)
    {
        $taskTypes = $this->getTaskTypeList();

        if (array_key_exists($id, $taskTypes) && $taskTypes[$id] instanceof TaskType)
        {
            return $taskTypes[$id];
        }

        $taskType = new TaskType();
        $taskType->name = 'Unknown Task Type';
        return $taskType;
    }

    /**
     * @return TourType[]|array
     */
    private function getTourTypeList()
    {
        if ($this->tourTypeList == null)
        {
            $this->tourTypeList = [];

            /**
             * @var $tourTypeTable TourTypesTable
             */
            $tourTypeTable = $this->serviceLocator->get('Application\Model\TourTypeTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $tourTypeTable->fetchAll();

            /**
             * @var $tourType TourType
             */
            while ($tourType = $resultSet->current())
            {
                $this->tourTypeList[$tourType->id] = $tourType;
                $resultSet->next();
            }
        }

        return $this->tourTypeList;
    }

    /**
     * @param $id
     * @return TourType
     */
    private function getTourType($id)
    {
        $tourTypes = $this->getTourTypeList();

        if (array_key_exists($id, $tourTypes) && $tourTypes[$id] instanceof TourType)
        {
            return $tourTypes[$id];
        }

        $tourType = new TourType();
        $tourType->name = 'Unknown Tour Type';
        return $tourType;
    }

    public function executeCronAction()
    {
        /**
         * @var $taskAssignmentTable TaskAssignmentTable
         */
        $taskAssignmentTable = $this->serviceLocator->get('Application\Model\TaskAssignmentTable');


        // $startDate = strtotime("+3 days");
        // $endDate   = strtotime("+9 days");

        $startDate = strtotime("now");


        /**
         * @var $taskAssignmentsResultSet ResultSet
         */
        $taskAssignmentsResultSet = $taskAssignmentTable->fetchCurrent($startDate, null, null, null);


        $failed_list = [];
        $completed_list = [];


        $guidesTasks = [];


        /**
         * @var $taskAssignment TaskAssignment
         */
        while ($taskAssignment = $taskAssignmentsResultSet->current())
        {
            try
            {
                /**
                 * @var $taskType TaskType
                 */
                $taskType = $this->getTaskType($taskAssignment->tasktype);

                /**
                 * @var $tourType TourType
                 */
                $tourType = $this->getTourType($taskAssignment->tourtype);


                $guidesAssigned = $this->getAssignedGuides($taskAssignment);

                /**
                 * If no guides are assigned, skip this task
                 */
                if (!is_array($guidesAssigned))
                {
                    $taskAssignmentsResultSet->next();
                    continue;
                }

                $taskData = new \stdClass();
                $taskData->taskAssignment = $taskAssignment;
                $taskData->taskType = $taskType;
                $taskData->tourType = $tourType;
                $taskData->assignedGuides = $guidesAssigned;

                foreach ($guidesAssigned as $guide)
                {
                    if (!array_key_exists($guide->id, $guidesTasks))
                    {
                        $guidesTasks[$guide->id] = [ 1=> [], 2 => []];
                    }
                    $guidesTasks[$guide->id][$taskAssignment->organization_id][] = $taskData;
                }

                $completed_list[] = $taskAssignment;
            } catch (\Exception $exception)
            {
                $failed_list[] = $taskAssignment;
            }

            $taskAssignmentsResultSet->next();
        }

        foreach ($guidesTasks as $guide_id => $tasks_list)
        {
            $user = $this->getUser($guide_id);

            $this->sendMail($user, [
                'start_date' => $startDate,
                // 'end_date'   => $endDate,
                'tasks'     => $tasks_list
            ]);
        }


        return [$completed_list, $failed_list];
    }

    /**
     * @param User|null $user
     * @param $mailData
     */
    public function sendMail(User $user = null, $mailData)
    {

        $config = $this->serviceLocator->get('config');
        $config = $config['smtp_service'];

        /**
         * @var $messageFooter EmailTemplate
         */
        $messageFooter = $this->serviceLocator->get('EmailFooterFactory');
        $messageFooter = $messageFooter->content;

        /**
         * @var $messageHeader EmailTemplate
         */
        $messageHeader = $this->serviceLocator->get('EmailHeaderFactory');
        $messageHeader = $messageHeader->content;


        $viewModel = new ViewModel();
        $viewModel->setTemplate('messaging/guide-future-schedule')
            ->setVariables([
                'start_date'    => $mailData['start_date'],
                // 'end_date'      => $mailData['end_date'],
                'tasks'         => $mailData['tasks'],
                'messageHeader' => $messageHeader,
                'messageFooter' => $messageFooter
            ]);


        $this->getMimeMessage(true)->addPart($this->getHtmlPart($viewModel, true));

        $mail = $this->getMailMessage(true);
        $mail->setFrom($config['default_from']);
        $mail->setTo($config['default_from']);

        // if (APPLICATION_ENVIRONMENT == 'production')
        // {
        //     $mail->setTo(sprintf("%s <%s>", $user->name, $user->email));
        //     $mail->addCc($config['default_from']);
        // } else
        // {
        //     $mail->setTo($config['debug_from']);
        // }

        // $mail->setSubject(sprintf("Future Schedule for %s-%s",
        //     date('l, F j, Y', $mailData['start_date']),
        //     date('l, F j, Y', $mailData['end_date'])
        // ));

        $mail->setSubject("Future Schedule for " .date('l, F j, Y', $mailData['start_date']));

        $mail->setBody($this->getMimeMessage());

        $this->dispatchMail();

    }
}
